'use client';

import { motion, useScroll, useTransform } from "motion/react"
import { useRef } from "react";

// 1. Improved Interface for type safety
interface LocalizedText {
    ar: string;
    en: string;
}

interface SectionProps {
    id: number;
    index: number;
    count: number;
    year: string;
    section: LocalizedText;
    title: LocalizedText;
    description: LocalizedText;
    locale: 'ar' | 'en';
}

export default function ItemSection(props: SectionProps) {
    const { count, id, year, section, title, description, locale, index } = props;

    // 2. Single reference for the whole container to trigger the animation
    const containerRef = useRef<HTMLDivElement>(null);

    const { scrollYProgress } = useScroll({
        target: containerRef,
        offset: ["start 80%", "start 30%"] // Adjusted offset for better visibility
    });

    const COLORS = {
        gray300: "#D1D5DB",
        mainBlue: "#1D305B",
        mainOrange: "#FF6400",
        white: "#ffffff"
    };

    const backgroundColor = useTransform(
        scrollYProgress,
        [0, 1],
        [COLORS.gray300, COLORS.mainOrange]
    );

    const textColor = useTransform(
        scrollYProgress,
        [0, 1],
        [COLORS.mainBlue, COLORS.white]
    );

    return (
        <div ref={containerRef} key={id} className="w-full min-h-fit flex items-center justify-center py-10">
            <div className="flex flex-col bg-gray-100 md:flex-row w-full max-w-5xl rounded-3xl min-h-45 relative shadow-sm">

                {/* Left Side: Image / Section Title */}
                <div
                    className="relative order-1 w-full md:w-69.5 min-h-45 md:min-h-auto bg-cover rounded-3xl md:rounded-s-3xl bg-center overflow-hidden"
                    style={{ backgroundImage: "url('/assets/images/00005.jpg')" }}
                >
                    <div className="absolute inset-0 bg-blue-900/60 mix-blend-multiply"></div>

                    <div className="absolute gap-10 flex flex-col items-center justify-center content-center bottom-6 right-1/2 translate-x-1/2 md:translate-x-0 md:left-auto md:right-8 z-10">
                        {/* Mobile Year Badge */}
                        <div className="md:hidden z-20 flex flex-col items-center">
                            <motion.div
                                style={{
                                    backgroundColor: backgroundColor,
                                    color: textColor
                                }}
                                className="relative z-10 font-bold text-xl py-2 px-8 rounded-full whitespace-nowrap"
                            >
                                {year}
                            </motion.div>
                        </div>
                        <h3 className="text-white font-bold text-4xl md:text-5xl tracking-wide">
                            {section[locale]}
                        </h3>
                    </div>
                </div>

                {/* Right Side: Description Content */}
                <div className="flex-1 p-8 md:p-12 flex flex-col justify-center order-2 relative">
                    {/* Desktop Year Badge & Line */}
                    <div className="hidden md:absolute md:top-10 md:-right-10 z-20 md:flex flex-col items-center">
                        <motion.div
                            style={{
                                backgroundColor: backgroundColor,
                                color: textColor
                            }}
                            className="relative z-10 font-bold text-3xl py-2 px-8 rounded-full whitespace-nowrap"
                        >
                            {year}
                        </motion.div>
                        
                        {/* Timeline Connector Line */}
                        {index + 1 !== count && (
                            <motion.div
                                style={{ backgroundColor: backgroundColor }}
                                className="hidden md:flex absolute top-12 w-1.5 h-64 z-0 origin-top"
                            />
                        )}
                    </div>

                    <h2 className="text-main-orange font-bold text-2xl md:text-3xl mb-4 text-right">
                        {title[locale]}
                    </h2>
                    <p className="text-gray-500 text-base leading-relaxed text-right max-w-2xl">
                        {description[locale]}
                    </p>
                </div>

            </div>
        </div>
    );
}